/*
 * mod_mcs.c: Codetable translation
 * 
 * Eugene G. Crosser <crosser@average.org>
 * 
 */

#include "httpd.h"
#include "http_config.h"

#include "mcs.h"

#include <stdarg.h>

#define MAX_TABNAME 3

typedef struct {
    array_header *langs;
    table *charsets;
    table *agents;
    array_header *suffixes;
    array_header *brokenuas;
    char *force;
} mcs_dir_conf;

module mcs_module;

/**************** Configuration handling *********************/

static void *create_mcs_config (pool *p, char *dummy)
{
    mcs_dir_conf *a =
      (mcs_dir_conf *)pcalloc (p, sizeof(mcs_dir_conf));

    a->langs = make_array (p, 4, sizeof(char*));
    a->charsets = make_table (p, 4);
    a->agents = make_table (p, 4);
    a->suffixes = make_array (p, 4, sizeof(char*));
    a->brokenuas = make_array (p, 4, sizeof(char*));
    a->force=NULL;
    return a;
}

static void *merge_mcs_config (pool *p, void *basev, void *addv)
{
    mcs_dir_conf *a =
	(mcs_dir_conf *)pcalloc (p, sizeof(mcs_dir_conf));
    mcs_dir_conf *base = (mcs_dir_conf *)basev,
	*add = (mcs_dir_conf *)addv;

    a->langs = append_arrays (p, add->langs, base->langs);
    a->charsets = overlay_tables (p, add->charsets, base->charsets);
    a->agents = overlay_tables (p, add->agents, base->agents);
    a->suffixes = append_arrays (p, add->suffixes, base->suffixes);
    a->brokenuas = append_arrays (p, add->brokenuas, base->brokenuas);
    if (add->force) a->force = add->force;
    return a;
}

static const char *add_tranlang(cmd_parms *cmd, mcs_dir_conf *s, char *l)
{
    char **a=(char**)push_array(s->langs);
    *a=pstrdup(cmd->pool,l);
    return NULL;
}

static const char *add_trancs(cmd_parms *cmd, mcs_dir_conf *s, char *f, char *r)
{
    table_set (s->charsets,r,f);
    return NULL;
}

static const char *add_tranua(cmd_parms *cmd, mcs_dir_conf *s, char *f, char *r)
{
    table_set (s->agents,r,f);
    return NULL;
}

static const char *add_transfx(cmd_parms *cmd, mcs_dir_conf *s, char *l)
{
    char **a=(char**)push_array(s->suffixes);
    *a=pstrdup(cmd->pool,l);
    return NULL;
}

static const char *add_brokenua(cmd_parms *cmd, mcs_dir_conf *s, char *l)
{
    char **a=(char**)push_array(s->brokenuas);
    *a=pstrdup(cmd->pool,l);
    return NULL;
}

static const char *add_force(cmd_parms *cmd, mcs_dir_conf *s, char *l)
{
    s->force=pstrdup(cmd->pool,l);
    return NULL;
}

static command_rec mcs_cmds[] = {
{ "TranslateLanguage", add_tranlang, NULL, OR_FILEINFO, ITERATE, 
    "languages that should be translated"},
{ "TranslateCharset", add_trancs, NULL, OR_FILEINFO, ITERATE2, 
    "translation name and charset names"},
{ "TranslateUserAgent", add_tranua, NULL, OR_FILEINFO, ITERATE2, 
    "translation name and user agent substrings"},
{ "TranslateSuffix", add_transfx, NULL, OR_FILEINFO, ITERATE, 
    "File suffixes to translate even if wrong language"},
{ "DefeatCharsetFor", add_brokenua, NULL, OR_FILEINFO, ITERATE, 
    "Substrings for user agents that do not like \"charset=\" attribute"},
{ "ForceTranslation", add_force, NULL, OR_FILEINFO, TAKE1, 
    "force pre-translation (usually directory-dependant)"},
{ NULL }
};

/********************** Translation ***********************/

extern void *(*btranslate)(void *buf, int nbytes);

static char *force=NULL;
static mcs_table_t *mcs_norm,*mcs_force;

/* We cannot translate in place because sometimes constant data (placed
   in the (read-only) code segment needs to be written out... */

static int bufsize=0;
static char *buffer=NULL;

static void *translate_buffer(void *buf, int nbytes) {
	if (buf == NULL) return buf;
	if (nbytes > bufsize) {
		free(buffer);
		bufsize=nbytes;
		buffer=(char*)malloc(bufsize);
	}
#ifdef MCS_DEBUG
	printf("will translate %d bytes at %08lx\n",nbytes,buf);
#endif
	
	while (--nbytes >= 0)
	    buffer[nbytes]=
		MCS_OUT(mcs_norm,MCS_IN(mcs_force,((char*)buf)[nbytes]));
#ifdef MCS_DEBUG
	printf("translated\n");
#endif
	return buffer;
}

/*********************** Handlers *************************/

static void initialize_mcs (server_rec *s, pool *p) {

    return;
}

/****************** Translation setup *********************/

int setup_translation (request_rec *r)
{
    char trname[4],*p;
    int i,do_translate=0;
    table *e = r->subprocess_env;
    mcs_dir_conf *conf =
	(mcs_dir_conf *)get_module_config(r->per_dir_config, &mcs_module);
    array_header *langs=conf->langs;
    table *charsets=conf->charsets;
    array_header *suffixes=conf->suffixes;
    array_header *brokenuas=conf->brokenuas;
    table *agents=conf->agents;
    char *acs=table_get(r->headers_in,"Accept-charset");
    table_entry *tte;
    char **tae;
    char *ccs=NULL;
    char *ua=table_get(r->headers_in,"User-Agent");

#ifdef MCS_DEBUG
    printf ("check: %s \"%s\",\"%s\" and reset (%s)\n",
		r->server->server_hostname,
		r->content_language?r->content_language:"(NULL)",
		r->content_type?r->content_type:"(NULL)",
		r->filename?r->filename:"(NULL)");
#endif
    /* reset translation */
    force=conf->force;
    btranslate=NULL;
    mcs_norm=mcstab_byname(NULL,NULL);
    if (force) {
	mcs_force=mcstab_byname(force,&i);
#ifdef MCS_DEBUG
	printf("set force table %s rc=%d\n",force,i);
#endif
    } else mcs_force=mcstab_byname(NULL,NULL);
    trname[0]='\0';

    if (force && (strncasecmp(r->content_type,"text",4) == 0))
	btranslate=translate_buffer;

    if (r->content_language) {
	tae=(char**)langs->elts;
	for (i=0;i<langs->nelts;i++) {
	    if (strcasecmp(tae[i],r->content_language) == 0) {
#ifdef MCS_DEBUG
		printf("allowed language: %s found\n",r->content_language);
#endif
		do_translate=1;
		break;
	    }
	}
    } else {
    /* Even .htm and .html should be translated sometimes.  Check suffixes */
	tae=(char**)suffixes->elts;
	if (p=strrchr(r->filename,'.')) p++;
	for (i=0;i<suffixes->nelts;i++) {
	    if (p && strcmp(tae[i],p) == 0) {
#ifdef MCS_DEBUG
		printf("Special suffix: %s found\n",p);
#endif
		/* r->content_language=pstrdup(r->pool,"ru"); */
		do_translate=1;
		break;
	    }
	}
    }

/* now we know if we want to translate the text at all */

    if (!do_translate) {
#ifdef MCS_DEBUG
	printf("No translation for \"%s\",\"%s\"\n",
		r->content_language?r->content_language:"(null)",
		r->content_type?r->content_type:"(null)");
#endif
	return DECLINED;
    }

    do_translate=0;

/* First, check if the user said what charset she wants explicitly */

    strncpy(trname,r->server->server_hostname,MAX_TABNAME);
    trname[MAX_TABNAME]='\0';
    if ((p=strchr(trname,'.'))) *p='\0';

    tte=(table_entry*)charsets->elts;
    for (i=0;i<charsets->nelts;i++) {
	if (strcasecmp(tte[i].val,trname) == 0) {
#ifdef MCS_DEBUG
	    printf("server name matches \"%s\", set charset \"%s\"\n",
			tte[i].val,tte[i].key);
#endif
	    do_translate=1;
	    ccs=tte[i].key;
	    break;
	}
    }

/* If no 'translation' hostname, check if there was "Accept-Charset:" */

    if (!do_translate && acs) {
	char *acs2,*best=NULL,*q;
	float bestq=0.0,qual;
#ifdef MCS_DEBUG
	printf("accept-charset: \"%s\"\n",acs);
#endif
	/* This is a pain.  We need to parse charset string and choose best
	   match according to q= parameter... */
	acs2=pstrdup(r->pool,acs);
	for (p=strtok(acs2,",");p;p=strtok(NULL,",")) {
	    qual=1.0;
	    if ((q=strchr(p,';'))) {
		*q++='\0';
		while (isspace(*q)) q++;
		if (strncasecmp(q,"q=",2) == 0) {
		    if (sscanf(q+2,"%g",&qual) != 1) qual=1.0;
#ifdef MCS_DEBUG
		    printf("q= parameter \"%s\" gave %4.2f\n",q,qual);
#endif
		}
	    }
	    while (isspace(*p)) p++;
	    q=p+strlen(p)-1;
	    while ((q > p) && isspace(*q)) *q--='\0';
#ifdef MCS_DEBUG
	    printf("we have charset \"%s\" with quality %4.2f\n",p,qual);
#endif
	    if ((qual > bestq) && table_get(conf->charsets,p)) {
		bestq=qual;
		best=p;
	    }
	}
#ifdef MCS_DEBUG
	printf("best charset: \"%s\" with quality %4.2f\n",
		best?best:"(null)",bestq);
#endif
	if (best) {
	    p=table_get(conf->charsets,best);
#ifdef MCS_DEBUG
	    printf("we chose charset \"%s\" with table \"%s\"\n",best,p);
#endif
	    strncpy(trname,p,MAX_TABNAME);
	    trname[3]='\0';
	    do_translate=1;
	    ccs=best;
	}
    }

/* As the last resort, try determine desired charset by the user agent */

    if (!do_translate) {
	if (ua) {
	    tte=(table_entry*)agents->elts;
#ifdef MCS_DEBUG
	    printf("check user-agent \"%s\"\n",ua);
#endif
		/* We cannot do simple table_get here because we want to
		   check strstr() match rather than strict match... */
	    for (i=0;i<agents->nelts;i++) {
		if (strstr(ua,tte[i].key)) {
#ifdef MCS_DEBUG
		    printf("user agent matches allowed table \"%s\"\n",
				tte[i].key);
#endif
		    do_translate=1;
		    strncpy(trname,tte[i].val,3);
		    trname[3]='\0';
		    break;
		}
	    }
	}
    }

    if (do_translate) {

	/* if we still do not know charset, take it from the table name */

	if (!ccs) {
	    tte=(table_entry*)charsets->elts;
	    for (i=0;i<charsets->nelts;i++) {
		if (strcasecmp(trname,tte[i].val) == 0) {
		    ccs=tte[i].key;
		    break;
		}
	    }
	}

	/* remove charset if the user agent is known to barf on it */

	if ((ccs != NULL) && (ua != NULL)) {
	    tae=(char**)brokenuas->elts;
#ifdef MCS_DEBUG
	    printf("check if user-agent \"%s\" is broken\n",ua);
#endif
		/* We cannot do simple table_get here because we want to
		   check strstr() match rather than strict match... */
	    for (i=0;i<brokenuas->nelts;i++) {
		if (strstr(ua,tae[i])) {
#ifdef MCS_DEBUG
		    printf("user agent \"%s\" is broken, remove charset\n",
				tae[i]);
#endif
		    ccs=NULL;
		    break;
		}
	    }
	}
#ifdef MCS_DEBUG
	printf("adding charset \"%s\"\n",ccs?ccs:"(null)");
#endif
	if (ccs) r->content_charset=pstrdup(r->pool,ccs);
	table_set (e, "LANG",trname);
	table_set (e, "CHARSET_TRANSLATION",trname);
#ifdef MCS_DEBUG
	printf("use \"%s\"\n",trname);
#endif
	mcs_norm=mcstab_byname(trname,&i);
#ifdef MCS_DEBUG
	printf("set normal table %s rc=%d\n",trname,i);
#endif
	btranslate=translate_buffer;
    }

    return DECLINED;
}

module mcs_module = {
   STANDARD_MODULE_STUFF,
   initialize_mcs,		/* initializer */
   create_mcs_config,		/* dir config creater */
   merge_mcs_config,		/* dir merger --- default is to override */
   NULL,			/* server config */
   NULL,			/* merge server configs */
   mcs_cmds,			/* command table */
   NULL,			/* handlers */
   NULL,			/* filename translation */
   NULL,			/* check_user_id */
   NULL,			/* check auth */
   NULL,			/* check access */
   NULL,			/* type_checker */
   setup_translation,		/* fixups */
   NULL				/* logger */
};
