#ifndef LINT
static char *rcsid="$Id: smscanner.c 323 2005-05-18 21:51:42Z crosser $";
#endif
                                                                                
/*
	WHAT IS IT:
		modularized contentfilter for Zmailer and Sendmail
	COPYRIGHT:
		(c) 2003-2005 Eugene G. Crosser <crosser@average.org>
	LICENSE:
		The same set as apply to Zmailer code
*/

/*
	developed using documentation from here:
		http://www.milter.org/milter_api/api.html
*/

#include "config.h"

#include <sys/types.h>
#ifdef STDC_HEADERS
# include <stdio.h>
# include <stdlib.h>
# include <stddef.h>
# include <string.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_ERRNO_H
# include <errno.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#include <signal.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/un.h>
#ifndef OPTARG_DEFINED
#include <getopt.h>
#endif

#include <libmilter/mfapi.h>

#include "rtconfig.h"
#include "report.h"
#include "zmscanner.h"
#include "daemon.h"

/* should be in some header file but I cannot find an appropriate one */
void init_plugin_hash(void);
void makepidfile(char *fn);

int verbose=0;
int isdaemon=1;

/* this frontend is only built in threaded version, no saveqfile anyway */
void saveqfile(void) {};

/* the thing does not work in milter:
RETSIGTYPE
terminate(int sig)
{
	DPRINT(("signal caught: %d\n",sig));
	(void)smfi_stop();
}
*/

/* Header scanners presume that text of the headers stay available for
   the whole time of message processing.  Content-Type, boundary, e.t.c.
   in the mattr structure point into the middle of the headers.
   Because milter does not give us the whole message, we'll have to keep
   all headers in a temporary chain.  It could be possible to only
   keep the chunks referenced by mattr, but that would require
   reimplementation of rfc822hdr scanner, and I don't want to touch
   code that is proved to work. */

typedef struct _hdr {
	struct _hdr *next;
	char text[4]; /* for ':', ' ', '\n' and trailing zero */
} hdr_t;

typedef struct _mlfiPriv
{
	struct _c {
		int local;
		int msgcount;
		char *peerstr;
		char *helostr;
	} c;

	struct _m {
		varpool_t vp;
		hdr_t *firsthdr;
		hdr_t *lasthdr;
		mattr_t matr;
		int freeze;
		int notfirstbodychunk;
	} m;
} mlfiPriv_t;

/* initialize connection part of priv strucure */
int
mlfi_priv_init_c(mlfiPriv_t *priv)
{
	priv->c.local=1;
	priv->c.msgcount=0;
	priv->c.peerstr=NULL;
	priv->c.helostr=NULL;
	return 0;
}

/* cleanup connection part of priv strucure */
void
mlfi_priv_cleanup_c(mlfiPriv_t *priv)
{
	if (priv->c.peerstr) free(priv->c.peerstr);
	priv->c.peerstr=NULL;
	if (priv->c.helostr) free(priv->c.helostr);
	priv->c.helostr=NULL;
}

/* initialize message part of priv strucure */
int
mlfi_priv_init_m(mlfiPriv_t *priv)
{
	if ((priv->m.vp=vp_create_vp()) == NULL) {
		ERRLOG((LOG_ERR,"vp_create_vp failed: %m"));
		return -1;
	}
	/* DPRINT(("mlfi_priv_init_m: varpool at %p\n",priv->vp)); */
	vp_set_str(priv->m.vp,VP_ANSWER_STR,"0 250 Acceptable");
	priv->m.firsthdr=NULL;
	priv->m.lasthdr=(hdr_t*)&(priv->m.firsthdr);
	initattr(&(priv->m.matr),NULL);
	priv->m.matr.usefirst=gcfg.firstmimehdr;
	priv->m.freeze=0;
	priv->m.notfirstbodychunk=0;
	return 0;
}

/* cleanup message part of priv strucure */
void
mlfi_priv_cleanup_m(mlfiPriv_t *priv)
{
	hdr_t *hdr;

	if (priv->m.vp) vp_destroy_vp(priv->m.vp);
	priv->m.vp=NULL;
	for (hdr=priv->m.firsthdr;hdr;) {
		hdr_t *prev=hdr;
		hdr=hdr->next;
		free(prev);
	}
	priv->m.firsthdr=NULL;
	priv->m.lasthdr=(hdr_t*)&(priv->m.firsthdr);
}

/* create new priv structure; return NULL if unsuccessfull */
static mlfiPriv_t *
mlfi_priv_new(SMFICTX *ctx)
{
	mlfiPriv_t *priv;

	if ((priv=malloc(sizeof *priv)) == NULL) {
		ERRLOG((LOG_ERR,"mlfi_priv_new: malloc priv: %m"));
		return NULL;
	}
	if (mlfi_priv_init_c(priv)) {
		ERRLOG((LOG_ERR,"mlfi_priv_new: init priv.c error"));
		mlfi_priv_cleanup_c(priv);
		return NULL;
	}
	if (mlfi_priv_init_m(priv)) {
		ERRLOG((LOG_ERR,"mlfi_priv_new: init priv.m error"));
		mlfi_priv_cleanup_m(priv);
		mlfi_priv_cleanup_c(priv);
		return NULL;
	}
	DPRINT(("mlfi_priv: created priv=%p\n",priv));
	return priv;
}

/* cleanup and free priv structure from ctx; set NULL in ctx */
void
mlfi_priv_destroy(SMFICTX *ctx)
{
	mlfiPriv_t *priv = (mlfiPriv_t *)smfi_getpriv(ctx);

	DPRINT(("mlfi_priv_destroy: priv=%p\n",priv));
	if (priv == NULL) return;
	mlfi_priv_cleanup_m(priv);
	mlfi_priv_cleanup_c(priv);
	free(priv);
	smfi_setpriv(ctx, NULL);
}

/* get priv from ctx; if not present then create and set into ctx */
static mlfiPriv_t *
mlfi_priv(SMFICTX *ctx)
{
	mlfiPriv_t *priv = (mlfiPriv_t *)smfi_getpriv(ctx);

	if (priv) {
		return priv;
	} else {
		priv=mlfi_priv_new(ctx);
		smfi_setpriv(ctx, priv);
		return priv;
	}
}

/* if scanner stopcode is not ZMSCAN_CONTINUE, get scanner's answer string,
   construct SMTP reply, calculate Milter disposition code and return it. */
static sfsistat
mlfi_termmode(SMFICTX *ctx,int stopcode)
{
	mlfiPriv_t *priv;
	char *ans;
	int rc;
	char rcode[4];
	char xcode[6];
	char *p,*q,*msg;

	if (stopcode == ZMSCAN_CONTINUE) return SMFIS_CONTINUE;
	priv=(mlfiPriv_t *)smfi_getpriv(ctx);
	if (priv == NULL) {
		ERRLOG((LOG_ERR,"mlfi_termmode: null priv, continue"));
		return SMFIS_CONTINUE;
	}
	ans=vp_get_str(priv->m.vp,VP_ANSWER_STR);
	if (ans == NULL) {
		ERRLOG((LOG_ERR,"mlfi_termmode: null answer, continue"));
		return SMFIS_CONTINUE;
	}

	rc=atoi(ans);
	p=strchr(ans,' ');
	if (p) while (*p == ' ') p++;
	if (p && (strspn(p,"0123456789") == 3)) strncpy(rcode,p,3);
	else strncpy(rcode,"500",3);
	rcode[3]='\0';
	if (p) q=strchr(p,' ');
	else q=NULL;
	if (q) while (*q == ' ') q++;
	if (q && (strspn(q,"0123456789.") == 5)) strncpy(xcode,q,5);
	else xcode[0]='\0';
	xcode[5]='\0';
	if (xcode[0]) msg=q+5;
	else if (p) msg=p+3;
	else msg=NULL;
	if (msg) while (*msg == ' ') msg++;
	DPRINT(("mlfi_termmode: rc=%d rcode=\"%s\" xcode=\"%s\"\n",
			rc,rcode,xcode?xcode:"(nil)"));
	(void)smfi_setreply(ctx,rcode,xcode[0]?xcode:NULL,msg);

	switch (rc) {
	case 0:
		return SMFIS_ACCEPT;
	case 1:
#ifdef SMFIF_QUARANTINE
		priv->m.freeze=1;
		return SMFIS_CONTINUE;
#endif /* SMFIF_QUARANTINE; otherwise act as if it was REJECT */
	case -1:
		switch (rcode[0]) {
		case '5':	return SMFIS_REJECT;
		case '4':	return SMFIS_TEMPFAIL;
		default:	ERRLOG((LOG_ERR,"stage returned \"%s\" which "
				"contains unknown rcode %s",ans,rcode));
				return SMFIS_ACCEPT;	/* or what? */
		}
	default: ERRLOG((LOG_ERR,"stage returned \"%s\" which "
				"contains unknown rc %d",ans,rc));
		return SMFIS_ACCEPT;	/* something bad */
	}
}

sfsistat
mlfi_conn(SMFICTX *ctx,char *hostname,_SOCK_ADDR *hostaddr)
{
	slab_t data;
	int rc;
	char buf[128];
	char caddr[64];
	int port;
	mlfiPriv_t *priv = mlfi_priv(ctx);

	DPRINT(("mlfi_conn: %s\n",hostname));
	if (priv == NULL) return SMFIS_TEMPFAIL;

	priv->c.local=0;
	switch (hostaddr->sa_family) {
	case AF_INET:
		inet_ntop(AF_INET,
			(void*)&((struct sockaddr_in*)hostaddr)->sin_addr,
			caddr,sizeof(caddr));
		port=((struct sockaddr_in*)hostaddr)->sin_port;
		break;
#ifdef AF_INET6
	case AF_INET6:
		inet_ntop(AF_INET6,
			(void*)&((struct sockaddr_in6*)hostaddr)->sin6_addr,
			caddr,sizeof(caddr));
		port=((struct sockaddr_in6*)hostaddr)->sin6_port;
		break;
#endif /* AF_INET6 */
	default:
		snprintf(caddr,sizeof(caddr),"-1 unknown family %d",
			hostaddr->sa_family);
		port=0;
	}
	snprintf(buf,sizeof(buf),"%s [%s]:%d",hostname,caddr,port);
	DPRINT(("peerdata is \"%s\"\n",buf));
	if (priv->c.peerstr == NULL)
		priv->c.peerstr=(char*)malloc(strlen(buf)+1);
	if (priv->c.peerstr) strcpy(priv->c.peerstr,buf);
	data.beg=buf;
	data.end=buf+strlen(buf);
	data.atr=(void *)&(priv->m.matr);
	rc=scandata("peerdata",0,data,priv->m.vp);
	return mlfi_termmode(ctx,rc);
}

sfsistat
mlfi_helo(SMFICTX *ctx,char *helohost)
{
	slab_t data;
	int rc;
	mlfiPriv_t *priv = mlfi_priv(ctx);

	DPRINT(("mlfi_helo: %s\n",helohost));
	if (priv == NULL) return SMFIS_TEMPFAIL;
	if (priv->c.local) return SMFIS_CONTINUE;
	if (priv->m.freeze) return SMFIS_CONTINUE;
	if (priv->c.helostr == NULL) {
		priv->c.helostr=(char*)malloc(strlen(helohost)+1);
	}
	if (priv->c.helostr) strcpy(priv->c.helostr,helohost);
	data.beg=helohost;
	data.end=helohost+strlen(helohost);
	data.atr=(void *)&(priv->m.matr);
	rc=scandata("helo",0,data,priv->m.vp);
	return mlfi_termmode(ctx,rc);
}

sfsistat
mlfi_envfrom(SMFICTX *ctx,char **argv)
{
	slab_t data;
	int rc;
	mlfiPriv_t *priv = mlfi_priv(ctx);

	DPRINT(("mlfi_envfrom: %s\n",argv[0]));
	if (priv == NULL) return SMFIS_TEMPFAIL;
	(priv->c.msgcount)++;
	if (priv->c.local) return SMFIS_CONTINUE;
	if (priv->c.msgcount > 1) {
		mlfi_priv_init_m(priv);
		if (priv->c.peerstr) {
			data.beg=priv->c.peerstr;
			data.end=(priv->c.peerstr)+strlen(priv->c.peerstr);
			data.atr=(void *)&(priv->m.matr);
			rc=scandata("peerdata",0,data,priv->m.vp);
			if (rc != ZMSCAN_CONTINUE)
				return mlfi_termmode(ctx,rc);
		}
		if (priv->c.helostr) {
			data.beg=priv->c.helostr;
			data.end=(priv->c.helostr)+strlen(priv->c.helostr);
			data.atr=(void *)&(priv->m.matr);
			rc=scandata("helo",0,data,priv->m.vp);
			if (rc != ZMSCAN_CONTINUE)
				return mlfi_termmode(ctx,rc);
		}
	}
	if (priv->m.freeze) return SMFIS_CONTINUE;
	data.beg=argv[0];
	data.end=argv[0]+strlen(argv[0]);
	data.atr=(void *)&(priv->m.matr);
	rc=scandata("envfrom",0,data,priv->m.vp);
	return mlfi_termmode(ctx,rc);
}

sfsistat
mlfi_envrcpt(SMFICTX *ctx,char **argv)
{
	slab_t data;
	int rc;
	mlfiPriv_t *priv = mlfi_priv(ctx);

	DPRINT(("mlfi_envrcpt: %s\n",argv[0]));
	if (priv == NULL) return SMFIS_TEMPFAIL;
	if (priv->c.local) return SMFIS_CONTINUE;
	if (priv->m.freeze) return SMFIS_CONTINUE;
	data.beg=argv[0];
	data.end=argv[0]+strlen(argv[0]);
	data.atr=(void *)&(priv->m.matr);
	rc=scandata("envrcpt",0,data,priv->m.vp);
	return mlfi_termmode(ctx,rc);
}

sfsistat
mlfi_header(SMFICTX *ctx,char *headerf,char *headerv)
{
	slab_t data;
	int rc;
	mlfiPriv_t *priv = mlfi_priv(ctx);
	hdr_t *hdr;

	DPRINT(("mlfi_header: %s: %s\n",headerf,headerv));
	if (priv == NULL) return SMFIS_TEMPFAIL;
	if (priv->c.local) return SMFIS_CONTINUE;
	if (priv->m.freeze) return SMFIS_CONTINUE;
	if ((hdr=malloc(sizeof *hdr+strlen(headerf)+strlen(headerv))) == NULL) {
		ERRLOG((LOG_ERR,"mlfi_conn: malloc error: %m"));
		mlfi_priv_destroy(ctx);
		return SMFIS_TEMPFAIL;
	}
	hdr->next=NULL;
	sprintf(hdr->text,"%s: %s\n",headerf,headerv);
	priv->m.lasthdr->next=hdr;
	priv->m.lasthdr=hdr;
	data.beg=hdr->text;
	data.end=hdr->text+strlen(hdr->text);
	data.atr=(void *)&(priv->m.matr);
	DPRINT(("header is \"%s\"\n",hdr->text));
	rc=scandata("rfc822hdr",0,data,priv->m.vp);
	return mlfi_termmode(ctx,rc);
}

sfsistat
mlfi_eoh(SMFICTX *ctx)
{
	mlfiPriv_t *priv = mlfi_priv(ctx);
	mattr_t *matr=&(priv->m.matr);

	DPRINT(("mlfi_eoh:\n"));
	/* matr should have been filled as a side effect of header scanning */
	DPRINT(("atr btype=%d\n",matr->btype));
	DPRINT(("atr benc=%d\n",matr->benc));
	DPRINT(("atr content_type=\"%.*s\"\n",
			slab_size(matr->content_type),matr->content_type.beg));
	DPRINT(("atr boundary=\"%.*s\"\n",
			slab_size(matr->boundary),matr->boundary.beg));
	DPRINT(("atr name=\"%.*s\"\n",
			slab_size(matr->name),matr->name.beg));
	DPRINT(("atr charset=\"%.*s\"\n",
			slab_size(matr->charset), matr->charset.beg));
	if ((matr->btype == t_multipart) && (slab_size(matr->boundary) == 0)) {
		/* I've seen DSNs from some broken MTA that generated
		   mulipart/mixed entities without boundary! Being
		   obvously broken, we have a choice of treating such
		   things as either text/plain or message/rfc822.
		   In my case, body of the entity was in fact an
		   enclosed message.  So, let's try this: */
		matr->btype=t_message;
	}

	return SMFIS_CONTINUE;
}

sfsistat
mlfi_body(SMFICTX *ctx,u_char *bodyp,size_t bodylen)
{
	slab_t data;
	int rc;
	mlfiPriv_t *priv = mlfi_priv(ctx);

	DPRINT(("mlfi_body: size %d, %sfirst entry\n",bodylen,
			priv->m.notfirstbodychunk?"not ":""));
	if (priv == NULL) return SMFIS_TEMPFAIL;
	if (priv->c.local) return SMFIS_CONTINUE;
	if (priv->m.freeze) return SMFIS_CONTINUE;
	if (priv->m.notfirstbodychunk) return SMFIS_CONTINUE;
	priv->m.notfirstbodychunk=1;
	data.beg=bodyp;
	data.end=bodyp+bodylen;
	data.atr=(void *)&(priv->m.matr);
	rc=scandata("body",0,data,priv->m.vp);
	return mlfi_termmode(ctx,rc);
}

sfsistat
mlfi_eom(SMFICTX *ctx)
{
	mlfiPriv_t *priv = mlfi_priv(ctx);
	sfsistat rc=SMFIS_CONTINUE;
	static char *nul="nul";
	slab_t data;
	strl_t *xhdr;

	DPRINT(("mlfi_eom\n"));
	if (priv == NULL) return SMFIS_TEMPFAIL;

	data.beg=nul;
	data.end=nul+strlen(nul);
	data.atr=(void *)&(priv->m.matr);
	(void)scandata(gcfg.finalstage,0,data,priv->m.vp);

	for (xhdr=vp_get_strl(priv->m.vp,VP_ADDHEADERS_STRL);
	     xhdr;
	     xhdr=xhdr->next) {
		char *p;

		DPRINT(("add header: %s\n",xhdr->text));
		p=strchr(xhdr->text,':');
		if (p) {
			ERRLOG((LOG_INFO,"Adding header \"%s\"",xhdr->text));
		} else {
			ERRLOG((LOG_ERR,"bad extra header \"%s\"",xhdr->text));
			continue;
		}
		*p++='\0'; /* damage it, but it'll be destroyed soon anyway */
		while ((*p == ' ') || (*p == '\t')) p++;
		if (smfi_addheader(ctx,xhdr->text,p)) {
			ERRLOG((LOG_ERR,"error adding header \"%s\" -> \"%s\"",
				xhdr->text,p));
		}
	}

#ifdef SMFIF_QUARANTINE
	if (priv->m.freeze) {
		mlfiPriv_t *priv = mlfi_priv(ctx);
		char *ans=vp_get_str(priv->m.vp,VP_ANSWER_STR);
		char *p=strchr(ans,' ');

		if (p) while (*p == ' ') p++;
		else p="Frozen";
		(void)smfi_quarantine(ctx,p);
		rc=SMFIS_ACCEPT; /* is this right?  or what? */
	} else rc=mlfi_termmode(ctx,ZMSCAN_STOP);
#endif /* SMFIF_QUARANTINE */

	mlfi_priv_cleanup_m(priv);
	return rc;
}

sfsistat
mlfi_abort(SMFICTX *ctx)
{
	mlfiPriv_t *priv = mlfi_priv(ctx);
	sfsistat rc=SMFIS_CONTINUE;

	DPRINT(("mlfi_abort\n"));
	/* return code irrelevant here,message already aborted */
	mlfi_priv_cleanup_m(priv);
	return rc;
}

sfsistat
mlfi_close(SMFICTX *ctx)
{
	mlfiPriv_t *priv = smfi_getpriv(ctx);

	DPRINT(("mlfi_close priv=%p\n",priv));
	mlfi_priv_destroy(ctx);
	return SMFIS_CONTINUE;
}

struct smfiDesc smfilter = {
	"smscanner",	/* filter name */
	SMFI_VERSION,	/* version code -- do not change */
	SMFIF_ADDHDRS
#ifdef SMFIF_QUARANTINE
	| SMFIF_QUARANTINE
#endif /* SMFIF_QUARANTINE */
		,	/* flags */
	mlfi_conn,	/* connection info filter */
	mlfi_helo,	/* SMTP HELO command filter */
	mlfi_envfrom,	/* envelope sender filter */
	mlfi_envrcpt,	/* envelope recipient filter */
	mlfi_header,	/* header filter */
	mlfi_eoh,	/* end of header */
	mlfi_body,	/* body block filter */
	mlfi_eom,	/* end of message */
	mlfi_abort,	/* message aborted */
	mlfi_close	/* connection cleanup */
};

static void
showver(char *name)
{
	fprintf(stderr,"%s Version %s Build %s\n",name,VERSION,__DATE__);
}

static void
showhelp(char *name)
{
	fprintf(stderr,"Usage: %s -I configfile -d -V -v -h\n",name);
	fprintf(stderr,"\t-I configfile\t- replacement config [default: %s]\n",DEFAULT_CONFIG);
	fprintf(stderr,"\t-p miltersock\t- replacement milter socket desc string\n");
	fprintf(stderr,"\t-d\t\t- do not go to background\n");
	fprintf(stderr,"\t-V\t\t- print version\n");
	fprintf(stderr,"\t-v\t\t- increase verbosity\n");
	fprintf(stderr,"\t-h\t\t- print this help message\n");
}

int
main(int argc, char *argv[])
{
	int i;
	int rc;
	char *configfile=DEFAULT_CONFIG;
	char *miltersock=NULL;

	while ((i=getopt(argc,argv,"I:p:dvVh")) != EOF) switch (i) {
	case 'I':
		/* if someone decides to modify the name displayed in
		   the "ps" output it may be useful to copy the arg... */
		configfile=(char *)malloc(strlen(optarg)+1);
		strcpy(configfile,optarg);
		break;
	case 'p':
		miltersock=(char *)malloc(strlen(optarg)+1);
		strcpy(miltersock,optarg);
		break;
	case 'd':
		isdaemon=0;
		break;
	case 'V':
		showver(argv[0]);
		return 0;
	case 'v':
		verbose++;
		break;
	case 'h':
		showhelp(argv[0]);
		return 0;
	default:
		showhelp(argv[0]);
		return 1;
	}

#ifdef HAVE_SYSLOG
	openlog("smscanner",LOG_CONS|LOG_PID
# ifdef LOG_NDELAY
					|LOG_NDELAY
# endif
						,LOG_MAIL);
#endif

	DPRINT(("process %d starting\n",getpid()));
	ERRLOG((LOG_INFO,"starting v.%s build %s %s",
					VERSION,__DATE__,__TIME__));
/* the thing does not work in milter:
	(void)signal(SIGTERM,terminate);
	(void)signal(SIGHUP,terminate);
	(void)signal(SIGINT,terminate);
*/
	init_plugin_hash();
	(void)readconfig(configfile,1);
	if (miltersock == NULL) miltersock=gcfg.miltersock;
	if (miltersock == NULL) {
		fprintf(stderr,"miltersock not specified\n");
		return 1;
	}

	if (isdaemon) {
		pid_t child;
		if ((child=fork()) > 0) {
			sleep(1);
			return 0;
		} else if (child < 0) {
			perror("fork error");
			return 1;
		} else {
			fflush(stdin);
			fflush(stdout);
			fflush(stderr);
			setbuf(stdin,NULL);
			setbuf(stdout,NULL);
			setbuf(stderr,NULL);
			close(0);
			close(1);
			close(2);
			open("/dev/null",O_RDWR);
			dup(0);
			dup(0);
#ifdef HAVE_SETSID
			setsid();
#else
# ifdef HAVE_SETPGRP
#  ifdef SETPGRP_VOID
			setpgrp();
#  else
			setpgrp(0,0);
#  endif
# endif
#endif
		}
	}

	if (smfi_setconn(miltersock) == MI_FAILURE) {
		ERRLOG((LOG_ERR,"smfi_setconn failed"));
		exit(1);
	}
	if (smfi_register(smfilter) == MI_FAILURE) {
		ERRLOG((LOG_ERR,"smfi_register failed"));
		exit(1);
	}
	makepidfile(gcfg.pidfile);
	init_plugins();
	init_dyn_modules();
	rc=smfi_main();
	term_plugins();
	DPRINT(("process %d terminating with rc=%d\n",getpid(),rc));
	ERRLOG((LOG_INFO,"process terminating rc=%d",rc));
#ifdef HAVE_SYSLOG
	closelog();
#endif
	return rc;
}
